/*
    Disk throughput benchmarking tool
    Copyright (C) 2006-2018 Darrick Wong

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#define PACKAGE "bogodisk"
#include "bogodisk.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <inttypes.h>
#include <unistd.h>
#include <stdint.h>
#include <fcntl.h>
#include <sched.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <sys/syscall.h>
#include <asm/unistd.h>
#include "util.h"

static int boost_ioprio = 1;
static int use_mlock = 1;
typedef ssize_t (*io_func_t)(int fd, void *buf, size_t count, off_t offset);
static io_func_t io_func = pread;
static int access_flag = O_RDONLY;
#ifdef O_DIRECT
static int direct_flag = O_DIRECT;
#else
static int direct_flag = 0;
#endif

static void print_help(void) {
	printf("Usage: %s [options] device [devices...] [-o report]\n", PACKAGE);
	printf("\n");
	printf("Options\n");
	printf(" -b	Start test at this location.\n");
	printf(" -c	Use memory buffer of this size.\n");
#ifdef O_DIRECT
	printf(" -d	Do not bypass page cache via O_DIRECT.\n");
#endif
	printf(" -e	End test after this location.\n");
	printf(" -i	Don't boost IO priority.\n");
	printf(" -o	Write output to the file \"report\".\n");
	printf(" -s	Number of samples to measure across the disk.  (0 = continuous.)\n");
	printf(" -w	Use destructive write test.\n");
	printf(" -y	Allow buffers to be paged out.\n");
	printf(" -z	Use O_SYNC.\n");
}

static void time_device(const char *dev, uint64_t start, uint64_t stop,
			uint64_t bufsize, FILE *report, uint64_t steps)
{
	uint64_t max_size, pos, interval = 0;
	void *buf;
	struct timespec now, before, rep;
	double nowf, beforef, repf;
	int clock_type = CLOCK_MONOTONIC;
	int fd, ret;
	uint32_t blksz;

	/* Flip backwards params */
	if (start && stop && (start > stop)) {
		max_size = start;
		start = stop;
		stop = max_size;
	}

	/* Check parameters */
	get_size_and_block_size(dev, &max_size, &blksz, NULL);
	if (!max_size) {
		fprintf(stderr, "%s: Size is zero, skipping.\n", dev);
		return;
	}

	if (start > max_size) {
		fprintf(stderr, "%s: Starting at %"PRIu64", which is beyond the "
			"end of the device at %"PRIu64"!\n", dev, start, max_size);
		return;
	}

	fprintf(stdout, "%s: Max size %"PRIu64"\n", dev, max_size);

	if (stop > max_size) {
		fprintf(stderr, "%s: Clamping end to %"PRIu64".\n", dev, max_size);
		stop = max_size;
	} else if (!stop) {
		stop = max_size;
	}
	
	/* Summarize operation */
	if (report != stdout)
		fprintf(stdout, "%s: %sing from %"PRIu64" to %"PRIu64
			" in %"PRIu64" byte chunks.\n", dev,
			(access_flag == O_RDONLY ? "Read" : "Writ"),
			start, stop, bufsize);
	fprintf(report, "%s: %sing from %"PRIu64" to %"PRIu64" in %"
		PRIu64" byte chunks.\n", dev,
		(access_flag == O_RDONLY ? "Read" : "Writ"),start, stop,
		bufsize);
	fflush(report);
	fflush(stdout);

	/* Get block size for alignment corrections */
	fprintf(stdout, "%s: Block size %d bytes.\n", dev, blksz);

	/* Calculate the stepping interval */
	if (steps)
		interval = (stop - start) / steps;
	if (direct_flag)
		interval &= ~(blksz - 1);

	/* grab buffer */
	if (posix_memalign(&buf, blksz, bufsize)) {
		perror("posix_memalign");
		return;
	}

	/* lock pages */
#ifdef _POSIX_MEMLOCK_RANGE
	if (use_mlock && mlock(buf, bufsize)) {
		perror("mlock");
		fprintf(stderr, "Can't lock pages; proceeding anyway.\n");
	}
#else
#warn mlock not present
#endif

	/* open device */
	fd = open(dev, access_flag | direct_flag | O_LARGEFILE);
	if (!fd) {
		perror("dev");
		goto unlock_mem;
	}

	/* make sure CLOCK_MONOTONIC is supported */
	if (clock_gettime(clock_type, &now)) {
		clock_type = CLOCK_REALTIME;
	}

	clock_gettime(clock_type, &rep);
	repf = (rep.tv_sec) + ((double)rep.tv_nsec / 1000000000);

	/* loop */
	pos = start;
	while (pos < stop) {
		clock_gettime(clock_type, &before);
		ret = io_func(fd, buf, bufsize, pos);
		if (!ret)
			break;
		else if (ret < 0) {
			perror(dev);
			if (ret != EIO)
				break;
			ret = 512;
		}
		clock_gettime(clock_type, &now);
		nowf = (now.tv_sec) + ((double)now.tv_nsec / 1000000000);
		beforef = (before.tv_sec) + ((double)before.tv_nsec / 1000000000);
		fprintf(report, "%"PRIu64", %"PRIu64", %.3f, %.2f\n", pos, pos + ret,
			(nowf - beforef),
			(double)ret / (nowf - beforef));
		if ((nowf - repf) > 5) {
			fprintf(stdout, "%s: Passed %"PRIu64
				" (%.2f%%)            \r",
				dev, pos + ret, (double)100 *
				(pos + ret - start) / (stop - start));
			fflush(stdout);
			repf = nowf;
		}
		if (steps)
			pos += interval;
		else
			pos += ret;
	}

	fprintf(stdout, "%s: Done.\n", dev);
	fflush(stdout);
	fflush(report);

	/* close device */
	close(fd);

unlock_mem:
	/* unlock pages */
#ifdef _POSIX_MEMLOCK_RANGE
	if (munlock(buf, bufsize)) {
		perror("munlock");
	}
#endif
	free(buf);
}

int main(int argc, char *argv[])
{
	uint64_t start, stop, bufsize, steps;
	int i;
	FILE *report = stdout;
	int c;
	int sync = 0;

	start = stop = steps = 0;
	bufsize = DEFAULT_BUFSIZE;

	fprintf(stdout, "%s %s, Copyright (C) 2006-2018 Darrick Wong.\n",
		PACKAGE, PACKAGE_VERSION);

	/* parse args */
	while((c = getopt(argc, argv, "dwb:e:c:o:s:iyz")) != -1) {
		switch(c) {
			case 'o':
				report = fopen(optarg, "w+");
				if (!report) {
					perror(optarg);
					return 2;
				}
				break;
			case 'b':
				start = get_number(optarg);
				break;
			case 'e':
				stop = get_number(optarg);
				break;
			case 'c':
				bufsize = get_number(optarg);
				break;
			case 'w':
				io_func = (io_func_t)pwrite;
				access_flag = O_WRONLY;
				break;
			case 'd':
				direct_flag = 0;
				break;
			case 's':
				steps = get_number(optarg);
				break;
			case 'z':
				sync = 1;
				break;
			case 'i':
				boost_ioprio = 0;
				break;
			case 'y':
				use_mlock = 0;
				break;
			default:
				print_help();
				return 1;
		}
	}
	if (sync)
		access_flag |= O_SYNC;

	if (argc == optind) {
		print_help();
		return 1;
	}

	if (boost_ioprio && bump_priority())
		return 2;

	for (i = optind; i < argc; i++)
		time_device(argv[i], start, stop, bufsize, report, steps);

	return 0;
}
